<?php
declare(strict_types=1);

require_once __DIR__ . '/config.php';

$pdo = iptv_pdo();
$ip = (string)($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0');

$username = trim((string)($_GET['username'] ?? ''));
$password = (string)($_GET['password'] ?? '');
$type = (string)($_GET['type'] ?? 'm3u_plus');
$output = (string)($_GET['output'] ?? 'ts');

$allowedTypes = ['m3u_plus'];
$allowedOutputs = ['ts', 'm3u8'];

if (!in_array($type, $allowedTypes, true) || !in_array($output, $allowedOutputs, true)) {
    http_response_code(400);
    header('Content-Type: text/plain; charset=utf-8');
    echo 'Invalid type/output.';
    exit;
}

$stmt = $pdo->prepare(
    'SELECT c.id AS client_id, c.username AS client_username, c.password_hash, c.active,
            s.id AS source_id, s.source_base_url, s.source_username, s.source_password_protected, s.default_type, s.default_output
     FROM clients c
     INNER JOIN sources s ON s.id = c.source_id
     WHERE c.username = :username
     LIMIT 1'
);
$stmt->execute([':username' => $username]);
$record = $stmt->fetch();

if (!$record || !password_verify($password, (string)$record['password_hash'])) {
    $clientId = $record ? (int)$record['client_id'] : null;
    iptv_log_access($pdo, $clientId, $ip, 401);
    http_response_code(401);
    header('Content-Type: text/plain; charset=utf-8');
    echo 'Unauthorized.';
    exit;
}

$clientId = (int)$record['client_id'];

if ((int)$record['active'] !== 1) {
    iptv_log_access($pdo, $clientId, $ip, 403);
    http_response_code(403);
    header('Content-Type: text/plain; charset=utf-8');
    echo 'Client inactive.';
    exit;
}

if (iptv_is_rate_limited($pdo, $ip, $clientId, 60, 60)) {
    iptv_log_access($pdo, $clientId, $ip, 429);
    http_response_code(429);
    header('Retry-After: 60');
    header('Content-Type: text/plain; charset=utf-8');
    echo 'Too many requests.';
    exit;
}

try {
    $sourcePassword = iptv_decrypt_secret((string)$record['source_password_protected']);
} catch (Throwable $e) {
    iptv_log_access($pdo, $clientId, $ip, 500);
    http_response_code(500);
    header('Content-Type: text/plain; charset=utf-8');
    echo 'Internal source credential error.';
    exit;
}

$sourceUrl = rtrim((string)$record['source_base_url'], '/') . '/get.php?' . http_build_query([
    'username' => (string)$record['source_username'],
    'password' => $sourcePassword,
    'type' => $type,
    'output' => $output,
]);

$ch = curl_init($sourceUrl);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HEADER => true,
    CURLOPT_CONNECTTIMEOUT => 5,
    CURLOPT_TIMEOUT => 20,
    CURLOPT_FOLLOWLOCATION => false,
    CURLOPT_USERAGENT => 'climener-proxy/1.0',
]);

$response = curl_exec($ch);
if ($response === false) {
    curl_close($ch);
    iptv_log_access($pdo, $clientId, $ip, 502);
    http_response_code(502);
    header('Content-Type: text/plain; charset=utf-8');
    echo 'Upstream source unreachable or timeout.';
    exit;
}

$httpCode = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
$headerSize = (int)curl_getinfo($ch, CURLINFO_HEADER_SIZE);
$rawHeaders = substr($response, 0, $headerSize);
$body = substr($response, $headerSize);
curl_close($ch);

if ($httpCode < 200 || $httpCode >= 300) {
    iptv_log_access($pdo, $clientId, $ip, 502);
    http_response_code(502);
    header('Content-Type: text/plain; charset=utf-8');
    echo 'Upstream source error.';
    exit;
}

$contentType = 'application/x-mpegURL';
if (preg_match('/^Content-Type:\s*([^\r\n]+)/mi', $rawHeaders, $m)) {
    $candidate = trim($m[1]);
    if ($candidate !== '') {
        $contentType = $candidate;
    }
}

iptv_log_access($pdo, $clientId, $ip, 200);

header('Content-Type: ' . $contentType);
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
echo $body;