<?php
declare(strict_types=1);

require_once __DIR__ . '/../config.php';
iptv_require_admin();

$pdo = iptv_pdo();
$error = '';
$success = '';
$generatedClient = null;

if (empty($_SESSION['iptv_csrf'])) {
    $_SESSION['iptv_csrf'] = bin2hex(random_bytes(16));
}

function generate_client_username(PDO $pdo): string
{
    for ($i = 0; $i < 10; $i++) {
        $candidate = 'cl' . bin2hex(random_bytes(4));
        $stmt = $pdo->prepare('SELECT id FROM clients WHERE username = :u LIMIT 1');
        $stmt->execute([':u' => $candidate]);
        if (!$stmt->fetch()) {
            return $candidate;
        }
    }
    throw new RuntimeException('Unable to generate unique client username.');
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $csrf = (string)($_POST['csrf'] ?? '');
    if (!hash_equals((string)$_SESSION['iptv_csrf'], $csrf)) {
        $error = 'CSRF token invalide.';
    } else {
        $sourceName = trim((string)($_POST['source_name'] ?? ''));
        $baseUrl = trim((string)($_POST['source_base_url'] ?? ''));
        $sourceUser = trim((string)($_POST['source_username'] ?? ''));
        $sourcePass = (string)($_POST['source_password'] ?? '');
        $defaultType = (string)($_POST['default_type'] ?? 'm3u_plus');
        $defaultOutput = (string)($_POST['default_output'] ?? 'ts');

        $allowedType = ['m3u_plus'];
        $allowedOutput = ['ts', 'm3u8'];

        if ($sourceName === '' || $baseUrl === '' || $sourceUser === '' || $sourcePass === '') {
            $error = 'Tous les champs source sont requis.';
        } elseif (!filter_var($baseUrl, FILTER_VALIDATE_URL)) {
            $error = 'source_base_url doit etre une URL valide.';
        } elseif (!in_array($defaultType, $allowedType, true)) {
            $error = 'default_type invalide.';
        } elseif (!in_array($defaultOutput, $allowedOutput, true)) {
            $error = 'default_output invalide.';
        } else {
            try {
                $encryptedSourcePassword = iptv_encrypt_secret($sourcePass);
                $clientUsername = generate_client_username($pdo);
                $clientPasswordPlain = bin2hex(random_bytes(6));
                $clientPasswordHash = password_hash($clientPasswordPlain, PASSWORD_BCRYPT);

                $pdo->beginTransaction();

                $stmtSource = $pdo->prepare(
                    'INSERT INTO sources (source_name, source_base_url, source_username, source_password_protected, default_type, default_output)
                     VALUES (:source_name, :base_url, :source_username, :source_password_protected, :default_type, :default_output)'
                );
                $stmtSource->execute([
                    ':source_name' => $sourceName,
                    ':base_url' => rtrim($baseUrl, '/'),
                    ':source_username' => $sourceUser,
                    ':source_password_protected' => $encryptedSourcePassword,
                    ':default_type' => $defaultType,
                    ':default_output' => $defaultOutput,
                ]);

                $sourceId = (int)$pdo->lastInsertId();

                $stmtClient = $pdo->prepare(
                    'INSERT INTO clients (username, password_hash, source_id, active)
                     VALUES (:username, :password_hash, :source_id, 1)'
                );
                $stmtClient->execute([
                    ':username' => $clientUsername,
                    ':password_hash' => $clientPasswordHash,
                    ':source_id' => $sourceId,
                ]);

                $clientId = (int)$pdo->lastInsertId();

                $pdo->commit();

                $generatedClient = [
                    'id' => $clientId,
                    'username' => $clientUsername,
                    'password' => $clientPasswordPlain,
                    'type' => $defaultType,
                    'output' => $defaultOutput,
                    'link' => iptv_client_url($clientUsername, $clientPasswordPlain, $defaultType, $defaultOutput),
                ];

                $success = 'Source et client crees avec succes.';
            } catch (Throwable $e) {
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                $error = 'Erreur lors de la creation: ' . $e->getMessage();
            }
        }
    }
}

$listStmt = $pdo->query(
    'SELECT s.id AS source_id, s.source_name, s.source_base_url, s.default_type, s.default_output, s.created_at,
            c.id AS client_id, c.username AS client_username, c.active
     FROM sources s
     LEFT JOIN clients c ON c.source_id = s.id
     ORDER BY s.id DESC'
);
$rows = $listStmt->fetchAll();
?>
<!doctype html>
<html lang="fr">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Admin Sources</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 0; background: #f7f7f8; color: #111; }
        .wrap { max-width: 1100px; margin: 24px auto; padding: 0 16px; }
        .top { display: flex; justify-content: space-between; align-items: center; }
        .btn { background: #1f2937; color: #fff; text-decoration: none; padding: 8px 12px; border-radius: 6px; }
        .card { background: #fff; border: 1px solid #ddd; border-radius: 8px; padding: 16px; margin-top: 16px; }
        .grid { display: grid; grid-template-columns: repeat(2, minmax(200px, 1fr)); gap: 12px; }
        label { display: block; font-size: 13px; margin-bottom: 4px; }
        input, select { width: 100%; box-sizing: border-box; padding: 9px; border: 1px solid #ccc; border-radius: 6px; }
        table { width: 100%; border-collapse: collapse; margin-top: 10px; }
        th, td { border-bottom: 1px solid #e6e6e6; padding: 8px; font-size: 14px; text-align: left; }
        .success { color: #065f46; background: #ecfdf5; border: 1px solid #a7f3d0; padding: 10px; border-radius: 6px; }
        .error { color: #991b1b; background: #fef2f2; border: 1px solid #fecaca; padding: 10px; border-radius: 6px; }
        .mono { font-family: Consolas, Menlo, monospace; font-size: 13px; }
        @media (max-width: 800px) { .grid { grid-template-columns: 1fr; } }
    </style>
</head>
<body>
<div class="wrap">
    <div class="top">
        <h1>Admin IPTV</h1>
        <a class="btn" href="/admin/login.php?logout=1">Logout</a>
    </div>

    <?php if ($success !== ''): ?>
        <div class="card success"><?= htmlspecialchars($success, ENT_QUOTES, 'UTF-8') ?></div>
    <?php endif; ?>

    <?php if ($error !== ''): ?>
        <div class="card error"><?= htmlspecialchars($error, ENT_QUOTES, 'UTF-8') ?></div>
    <?php endif; ?>

    <?php if ($generatedClient !== null): ?>
        <div class="card">
            <h3>Acces client genere</h3>
            <p class="mono">client_username: <?= htmlspecialchars($generatedClient['username'], ENT_QUOTES, 'UTF-8') ?></p>
            <p class="mono">client_password: <?= htmlspecialchars($generatedClient['password'], ENT_QUOTES, 'UTF-8') ?></p>
            <p class="mono">lien final: <?= htmlspecialchars($generatedClient['link'], ENT_QUOTES, 'UTF-8') ?></p>
        </div>
    <?php endif; ?>

    <div class="card">
        <h2>Ajouter une source</h2>
        <form method="post" autocomplete="off">
            <input type="hidden" name="csrf" value="<?= htmlspecialchars((string)$_SESSION['iptv_csrf'], ENT_QUOTES, 'UTF-8') ?>">
            <div class="grid">
                <div>
                    <label for="source_name">source_name</label>
                    <input id="source_name" name="source_name" required>
                </div>
                <div>
                    <label for="source_base_url">source_base_url</label>
                    <input id="source_base_url" name="source_base_url" value="http://" required>
                </div>
                <div>
                    <label for="source_username">source_username</label>
                    <input id="source_username" name="source_username" required>
                </div>
                <div>
                    <label for="source_password">source_password</label>
                    <input id="source_password" type="password" name="source_password" required>
                </div>
                <div>
                    <label for="default_type">default_type</label>
                    <select id="default_type" name="default_type">
                        <option value="m3u_plus">m3u_plus</option>
                    </select>
                </div>
                <div>
                    <label for="default_output">default_output</label>
                    <select id="default_output" name="default_output">
                        <option value="ts">ts</option>
                        <option value="m3u8">m3u8</option>
                    </select>
                </div>
            </div>
            <p><button class="btn" type="submit">Ajouter source + generer client</button></p>
        </form>
    </div>

    <div class="card">
        <h2>Sources et clients</h2>
        <table>
            <thead>
                <tr>
                    <th>ID source</th>
                    <th>source_name</th>
                    <th>source_base_url</th>
                    <th>client_username</th>
                    <th>active</th>
                    <th>created_at</th>
                </tr>
            </thead>
            <tbody>
            <?php foreach ($rows as $row): ?>
                <tr>
                    <td><?= (int)$row['source_id'] ?></td>
                    <td><?= htmlspecialchars((string)$row['source_name'], ENT_QUOTES, 'UTF-8') ?></td>
                    <td class="mono"><?= htmlspecialchars((string)$row['source_base_url'], ENT_QUOTES, 'UTF-8') ?></td>
                    <td class="mono"><?= htmlspecialchars((string)($row['client_username'] ?? ''), ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= ((int)($row['active'] ?? 0) === 1) ? 'yes' : 'no' ?></td>
                    <td><?= htmlspecialchars((string)$row['created_at'], ENT_QUOTES, 'UTF-8') ?></td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>
</body>
</html>